/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __RTSP_MESSAGE_H__
#define __RTSP_MESSAGE_H__

/**
 * @file rtsp_message.h
 * @brief RTSP Message Definition
 */ 

#include "rtsp_start_line.h"
#include "rtsp_header.h"

APT_BEGIN_EXTERN_C

typedef struct rtsp_message_t rtsp_message_t;

/** RTSP message */
struct rtsp_message_t {
	/** RTSP mesage type (request/response) */
	rtsp_start_line_t start_line;     
	/** RTSP header */
	rtsp_header_t     header;
	/** RTSP message body */
	char             *body;

	apr_pool_t       *pool;
};

/** Initialize RTSP message */
static APR_INLINE void rtsp_message_init(rtsp_message_t *message, rtsp_message_type message_type, apr_pool_t *pool)
{
	message->pool = pool;
	rtsp_start_line_init(&message->start_line,message_type);
	rtsp_header_init(&message->header);
	message->body = NULL;
}

/** Create RTSP message */
static APR_INLINE rtsp_message_t* rtsp_message_create(rtsp_message_type message_type, apr_pool_t *pool)
{
	rtsp_message_t *message = apr_palloc(pool,sizeof(rtsp_message_t));
	rtsp_message_init(message,message_type,pool);
	return message;
}

/** 
 * Create RTSP request message.
 * @param engine the engine
 */
rtsp_message_t* rtsp_request_create(apr_pool_t *pool);

/** 
 * Create RTSP response message.
 * @param engine the engine
 * @param request the request to create response to
 * @param status_code the status code of the response
 * @param reason the reason phrase of the response
 */
rtsp_message_t* rtsp_response_create(rtsp_message_t *request, rtsp_status_code_t status_code, const char *reason, apr_pool_t *pool);

/** Destroy MRCP message */
static APR_INLINE void rtsp_message_destroy(rtsp_message_t *message)
{
	/* nothing to do message is allocated from pool */
}

/** Parse RTSP message */
apt_bool_t rtsp_message_parse(rtsp_message_t *message, apt_text_stream_t *text_stream);

/** Generate RTSP message */
apt_bool_t rtsp_message_generate(rtsp_message_t *message, apt_text_stream_t *text_stream);


APT_END_EXTERN_C

#endif /*__RTSP_MESSAGE_H__*/
