/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_RESOURCE_H__
#define __MRCP_RESOURCE_H__

/**
 * @file mrcp_resource.h
 * @brief Abstract MRCP Resource
 */ 

#include "mrcp_header_base.h"

MRCP_BEGIN_EXTERN_C

/** Audio stream type definition (source/sink) needed for MRCP resource to stream audio */
typedef enum {
	MRCP_AUDIO_STREAM_NONE,
	MRCP_AUDIO_STREAM_SOURCE,
	MRCP_AUDIO_STREAM_SINK
} mrcp_audio_stream_type;

/** MRCP resource definition */
struct mrcp_resource_t {
	/** MRCP resource identifier */
	mrcp_resource_id                id;
	/** MRCP resource name */
	const char                     *name;

	/** MRCP methods registered in the resource */
	const apt_string_table_item_t *method_table;
	/** Number of MRCP methods registered in the resource */
	size_t                          method_count;

	/** MRCP events registered in the resource */
	const apt_string_table_item_t *event_table;
	/** Number of MRCP events registered in the resource */
	size_t                         event_count;

	/** Audio stream type needed for the resource */
	mrcp_audio_stream_type         audio_stream_type;

	/** MRCP resource header accessor */
	mrcp_header_accessor_t         header_accessor;
	/** MRCP generic header accessor (common/shared among the resources) */
	mrcp_header_accessor_t        *generic_header_accessor;
};

/** Initializes MRCP resource */
static APR_INLINE void mrcp_resource_init(mrcp_resource_t *resource)
{
	resource->method_count = 0;
	resource->method_table = NULL;
	resource->event_count = 0;
	resource->event_table = NULL;
	resource->audio_stream_type = MRCP_AUDIO_STREAM_NONE;

	mrcp_header_accessor_init(&resource->header_accessor);
}

/** Validates MRCP resource */
static APR_INLINE mrcp_status_t mrcp_resource_validate(mrcp_resource_t *resource)
{
	return (resource->name && resource->method_table && resource->method_count && 
		resource->event_table && resource->event_count && 
		mrcp_header_accessor_validate(&resource->header_accessor) == MRCP_STATUS_SUCCESS &&
		resource->generic_header_accessor) ? MRCP_STATUS_SUCCESS : MRCP_STATUS_FAILURE;
}

/** Resource container definition, which stores available MRCP resources */
struct mrcp_resource_container_t {
	/** Array of MRCP resources */
	mrcp_resource_t              **resource_array;
	/** Number of MRCP resources */
	size_t                         resource_count;
	/** String table of MRCP resources */
	const apt_string_table_item_t *resource_string_table;

	/** MRCP generic header accessor (common/shared among the resources) */
	mrcp_header_accessor_t         generic_header_accessor;
};

/** Create MRCP resource container */
mrcp_resource_container_t* mrcp_resource_container_create(size_t resource_count, const apt_string_table_item_t *resource_string_table, apr_pool_t *pool);

/** Destroy MRCP resource container */
mrcp_status_t mrcp_resource_container_destroy(mrcp_resource_container_t *resource_container);

/** Register MRCP resource */
mrcp_status_t mrcp_resource_register(mrcp_resource_container_t *resource_container, mrcp_resource_t *resource);


/** Get resource name associated with specified resource id */
static APR_INLINE const char* mrcp_resource_name_get(mrcp_resource_container_t *resource_container, mrcp_resource_id resource_id)
{
	return apt_string_table_get(resource_container->resource_string_table,resource_container->resource_count,resource_id);
}

/** Find resource id associated with specified resource name */
static APR_INLINE mrcp_resource_id mrcp_resource_id_find(mrcp_resource_container_t *resource_container, const char *resource_name)
{
	return apt_string_table_find(resource_container->resource_string_table,resource_container->resource_count,resource_name);
}

MRCP_END_EXTERN_C

#endif /*__MRCP_RESOURCE_H__*/
