/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_MEDIA_CONTROL_H__
#define __MRCP_MEDIA_CONTROL_H__

/**
 * @file mrcp_media_control.h
 * @brief MRCP Control Media (SDP m=application)
 */ 

#include "mrcp_media_base.h"

MRCP_BEGIN_EXTERN_C

/** MRCP proto transport (v2) */
typedef enum {
	MRCP_PROTO_TCP,
	MRCP_PROTO_TLS,

	MRCP_PROTO_COUNT,
	MRCP_PROTO_UNKNOWN = MRCP_PROTO_COUNT
}mrcp_proto_type;


/** MRCP control attributes */
typedef enum {
	MRCP_CONTROL_ATTRIB_SETUP,
	MRCP_CONTROL_ATTRIB_CONNECTION,
	MRCP_CONTROL_ATTRIB_RESOURCE,
	MRCP_CONTROL_ATTRIB_CHANNEL,
	MRCP_CONTROL_ATTRIB_CMID,

	MRCP_CONTROL_ATTRIB_COUNT,
	MRCP_CONTROL_ATTRIB_UNKNOWN = MRCP_CONTROL_ATTRIB_COUNT
}mrcp_control_attrib_id;


/** MRCP setup attributes */
typedef enum {
	MRCP_CONTROL_SETUP_TYPE_ACTIVE,
	MRCP_CONTROL_SETUP_TYPE_PASSIVE,

	MRCP_CONTROL_SETUP_TYPE_COUNT,
	MRCP_CONTROL_SETUP_TYPE_UNKNOWN = MRCP_CONTROL_SETUP_TYPE_COUNT
} mrcp_control_setup_type;

/** MRCP connection attributes */
typedef enum {
	MRCP_CONTROL_CONNECTION_TYPE_NEW,
	MRCP_CONTROL_CONNECTION_TYPE_EXISTING,

	MRCP_CONTROL_CONNECTION_TYPE_COUNT,
	MRCP_CONTROL_CONNECTION_TYPE_UNKNOWN = MRCP_CONTROL_CONNECTION_TYPE_COUNT
} mrcp_control_connection_type;


typedef struct mrcp_media_control_t mrcp_media_control_t;
/** MRCP control media */
struct mrcp_media_control_t {
	mrcp_media_base_t            base;

	mrcp_proto_type              proto;
	mrcp_control_setup_type      setup_type;
	mrcp_control_connection_type connection_type;
	const char                  *resource_name;
	mrcp_session_id              session_id;
	size_t                       cmid;
};

/** Initialize MRCP control media */
static APR_INLINE void mrcp_media_control_init(mrcp_media_control_t *media)
{
	mrcp_media_base_init(&media->base);
	media->proto = MRCP_PROTO_UNKNOWN;
	media->setup_type = MRCP_CONTROL_SETUP_TYPE_UNKNOWN;
	media->connection_type = MRCP_CONTROL_CONNECTION_TYPE_UNKNOWN;
	media->resource_name = NULL;
	media->session_id.hex_str = NULL;
	media->session_id.length = 0;
	media->cmid = 0;
}

/** Get MRCP protocol transport name by identifier */
const char* mrcp_proto_get(mrcp_proto_type proto);

/** Find MRCP protocol transport identifier by name */
mrcp_proto_type mrcp_proto_find(const char *attrib);


/** Get MRCP control attribute name by identifier */
const char* mrcp_control_attrib_str_get(mrcp_control_attrib_id attrib_id);

/** Find MRCP control attribute identifier by name */
mrcp_control_attrib_id mrcp_control_attrib_id_find(const char *attrib);


/** Get MRCP control setup type name by identifier */
const char* mrcp_control_setup_type_get(mrcp_control_setup_type setup_type);

/** Find MRCP control setup type identifier by name */
mrcp_control_setup_type mrcp_control_setup_type_find(const char *attrib);


/** Get MRCP control connection type name by identifier */
const char* mrcp_control_connection_type_get(mrcp_control_connection_type connection_type);

/** Find MRCP control connection type identifier by name */
mrcp_control_connection_type mrcp_control_connection_type_find(const char *attrib);


MRCP_END_EXTERN_C

#endif /*__MRCP_MEDIA_CONTROL_H__*/
