/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __APT_PTR_LIST_H__
#define __APT_PTR_LIST_H__

/**
 * @file apt_ptr_list.h
 * @brief List of Opaque void*
 */ 

#include "apt.h"

APT_BEGIN_EXTERN_C


typedef struct apt_ptr_list_t apt_ptr_list_t;
typedef struct apt_list_elem_t apt_list_elem_t;

/**
 * Create list.
 * @param pool the pool to allocate list from
 * @return the created list
 */
APT_DECLARE(apt_ptr_list_t*) apt_list_create(apr_pool_t *pool);

/**
 * Destroy list.
 * @param list the list to destroy
 */
APT_DECLARE(void) apt_list_destroy(apt_ptr_list_t *list);

/**
 * Push data to the list as first in, first out.
 * @param list the list to push data to
 * @param data the data to push
 * @return the inserted element
 */
APT_DECLARE(apt_list_elem_t*) apt_list_push_back(apt_ptr_list_t *list, void *data);

/**
 * Pop data from the list as first in, first out.
 * @param list the list to pop message from
 * @return the popped message (if any)
 */
APT_DECLARE(void*) apt_list_pop_front(apt_ptr_list_t *list);

/**
 * Retrieve data of the first element in the list.
 * @param list the list to retrieve from
 */
APT_DECLARE(void*) apt_list_head(apt_ptr_list_t *list);

/**
 * Retrieve data of the last element in the list.
 * @param list the list to retrieve from
 */
APT_DECLARE(void*) apt_ptr_list_tail(apt_ptr_list_t *list);


/**
 * Retrieve the first element of the list.
 * @param list the list to retrieve from
 */
APT_DECLARE(apt_list_elem_t*) apt_list_first_elem_get(apt_ptr_list_t *list);

/**
 * Retrieve the last element of the list.
 * @param list the list to retrieve from
 */
APT_DECLARE(apt_list_elem_t*) apt_list_last_elem_get(apt_ptr_list_t *list);

/**
 * Retrieve the next element of the list.
 * @param list the list to retrieve from
 * @param elem the element to retrieve next element from
 */
APT_DECLARE(apt_list_elem_t*) apt_list_next_elem_get(apt_ptr_list_t *list, apt_list_elem_t *elem);

/**
 * Retrieve the prev element of the list.
 * @param list the list to retrieve from
 * @param elem the element to retrieve prev element from
 */
APT_DECLARE(apt_list_elem_t*) apt_list_prev_elem_get(apt_ptr_list_t *list, apt_list_elem_t *elem);

/**
 * Insert element to the list.
 * @param list the list to insert element to
 * @param elem the element to insert before
 * @param data the data to insert
 * @return the inserted element
 */
APT_DECLARE(apt_list_elem_t*) apt_list_elem_insert(apt_ptr_list_t *list, apt_list_elem_t *elem, void *data);

/**
 * Remove element from the list.
 * @param list the list to remove element from
 * @param elem the element to remove
 * @return the next element (if any)
 */
APT_DECLARE(apt_list_elem_t*) apt_list_elem_remove(apt_ptr_list_t *list, apt_list_elem_t *elem);

/**
 * Retrieve the data associated with element.
 * @param elem the element to retrieve data from
 */
APT_DECLARE(void*) apt_list_elem_data_get(apt_list_elem_t *elem);


APT_END_EXTERN_C

#endif /*__APT_PTR_LIST_H__*/
