/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include "mrcp_resource_manager.h"
#include "mrcp_parser.h"
#include "mrcp_generic_header.h"
#include "mrcp_synthesizer.h"
#include "mrcp_resource_set.h"

mrcp_status_t parse_generate_test(apt_text_stream_t *parse_stream, apt_text_stream_t *generate_stream, const char *resource_name, apr_pool_t *pool)
{
	mrcp_message_t mrcp_message;
	mrcp_resource_container_t *resource_container;

	mrcp_version_t version = MRCP_VERSION_2;
	if(resource_name) {
		/* resource name is required only for v1 */
		version = MRCP_VERSION_1;
	}

	apt_log(APT_PRIO_INFO,"1) Parse and Generate\n");
	resource_container = mrcp_parser_resource_container_create(version,NULL,pool);
	if(!resource_container) {
		apt_log(APT_PRIO_WARNING,"failed to create resource container\n");
		return MRCP_STATUS_FAILURE;
	}

	apt_log(APT_PRIO_INFO,"-->PARSE\n");
	apt_log(APT_PRIO_INFO,parse_stream->buffer);
	mrcp_message_init(&mrcp_message,pool);
	if(resource_name) {
		mrcp_message.channel_id.resource_name = resource_name;
	}
	if(mrcp_message_parse(resource_container,&mrcp_message,parse_stream) == MRCP_STATUS_FAILURE) {
		apt_log(APT_PRIO_WARNING,"failed to parse message\n");
		return MRCP_STATUS_FAILURE;
	}

	mrcp_message.start_line.length = 0;

	if(mrcp_message_generate(resource_container,&mrcp_message,generate_stream) == MRCP_STATUS_FAILURE) {
		apt_log(APT_PRIO_WARNING,"failed to generate message\n");
		return MRCP_STATUS_FAILURE;
	}
	*generate_stream->pos = '\0';
	mrcp_message_destroy(&mrcp_message);
	apt_log(APT_PRIO_INFO,"\n-->GENERATE\n");
	apt_log(APT_PRIO_INFO,generate_stream->buffer);

	mrcp_resource_container_destroy(resource_container);
	return MRCP_STATUS_SUCCESS;
}

mrcp_status_t generate_parse_test(apt_text_stream_t *text_stream, apr_pool_t *pool)
{
	mrcp_message_t mrcp_message;
	mrcp_generic_header_t *generic_header;
	mrcp_synthesizer_header_t *synthesizer_header;
	mrcp_resource_container_t *resource_container;
	mrcp_version_t version = MRCP_VERSION_2;

	apt_log(APT_PRIO_INFO,"\n2) Generate and Parse\n");
	resource_container = mrcp_parser_resource_container_create(version,NULL,pool);
	if(!resource_container) {
		apt_log(APT_PRIO_WARNING,"failed to create resource container\n");
		return MRCP_STATUS_FAILURE;
	}

	mrcp_message_init(&mrcp_message,pool);
	mrcp_message.start_line.version = MRCP_VERSION_2;
	mrcp_message.start_line.message_type = MRCP_MESSAGE_TYPE_REQUEST;
	mrcp_message.start_line.method_id = SYNTHESIZER_SPEAK;
	mrcp_message.channel_id.resource_id = MRCP_RESOURCE_SYNTHESIZER;
	if(mrcp_message_associate_resource_by_id(resource_container,&mrcp_message) != MRCP_STATUS_SUCCESS) {
		apt_log(APT_PRIO_WARNING,"failed to associate resource\n");
		return MRCP_STATUS_FAILURE;
	}

	mrcp_message.start_line.request_id = 4567;
	mrcp_message.channel_id.session_id.hex_str = "12345678";
	mrcp_message.channel_id.session_id.length = strlen(mrcp_message.channel_id.session_id.hex_str);

	synthesizer_header = mrcp_resource_header_prepare(&mrcp_message);
	generic_header = mrcp_generic_header_prepare(&mrcp_message);

	if(!synthesizer_header || !generic_header) {
		apt_log(APT_PRIO_WARNING,"failed to allocate headers\n");
		return MRCP_STATUS_FAILURE;
	}
	
	synthesizer_header->voice_param.gender = VOICE_GENDER_NEUTRAL;
	mrcp_resource_header_property_add(&mrcp_message,SYNTHESIZER_HEADER_VOICE_GENDER);
	synthesizer_header->voice_param.age = 25;
	mrcp_resource_header_property_add(&mrcp_message,SYNTHESIZER_HEADER_VOICE_AGE);
	synthesizer_header->prosody_param.volume = PROSODY_VOLUME_MEDIUM;
	mrcp_resource_header_property_add(&mrcp_message,SYNTHESIZER_HEADER_PROSODY_VOLUME);

	mrcp_message.body = apr_pstrdup(pool,"Hello World");
	
	generic_header->content_type = apr_pstrdup(pool,"text");
	mrcp_generic_header_property_add(&mrcp_message,GENERIC_HEADER_CONTENT_TYPE);
	generic_header->content_length = strlen(mrcp_message.body);
	mrcp_generic_header_property_add(&mrcp_message,GENERIC_HEADER_CONTENT_LENGTH);

	if(mrcp_message_generate(resource_container,&mrcp_message,text_stream) == MRCP_STATUS_FAILURE) {
		apt_log(APT_PRIO_WARNING,"failed to generate message\n");
		return MRCP_STATUS_FAILURE;
	}
	*text_stream->pos = '\0';
	apt_log(APT_PRIO_INFO,"-->GENERATE\n");
	apt_log(APT_PRIO_INFO,text_stream->buffer);

	{
		mrcp_message_t mrcp_message2;
		text_stream->pos = text_stream->buffer;
		apt_log(APT_PRIO_INFO,"\n-->PARSE\n");
		mrcp_message_init(&mrcp_message2,pool);
		if(mrcp_message_parse(resource_container,&mrcp_message2,text_stream) == MRCP_STATUS_FAILURE) {
			apt_log(APT_PRIO_WARNING,"failed to parse message\n");
			return MRCP_STATUS_FAILURE;
		}
		
		/* mrcp_message and mrcp_message2 should be compared here */

		mrcp_message_destroy(&mrcp_message2);
	}

	mrcp_message_destroy(&mrcp_message);
	mrcp_resource_container_destroy(resource_container);
	return MRCP_STATUS_SUCCESS;
}

#define TEST_BUFFER_SIZE 2048

int main(int argc, char *argv[])
{
	char parse_buffer[TEST_BUFFER_SIZE];
	char generate_buffer[TEST_BUFFER_SIZE];
	apt_text_stream_t parse_stream;
	apt_text_stream_t generate_stream;
	const char *resource_name = NULL;
	apr_pool_t *pool;

	FILE *f;
	if(argc < 2) {
		printf("usage: parsertest mrcp.msg [resource name]\n");
		return 0;
	}
	if(argc >= 3) {
		resource_name = argv[2];
	}
	f = fopen(argv[1], "rb");
	if(f == NULL) {
		printf("cannot open file %s\n", argv[1]);
		return 0;
	}

	if(apr_initialize() != APR_SUCCESS) {
		fclose(f);
		apr_terminate();
		return 0;
	}

	if(apr_pool_create(&pool,NULL) != APR_SUCCESS) {
		fclose(f);
		apr_terminate();
		return 0;
	}

	parse_stream.size = fread(parse_buffer, 1, sizeof(parse_buffer)-1, f);
	parse_buffer[parse_stream.size] = '\0';
	parse_stream.buffer = parse_buffer;
	parse_stream.pos = parse_buffer;

	generate_stream.buffer = generate_buffer;
	generate_stream.pos = generate_buffer;
	generate_stream.size = TEST_BUFFER_SIZE;

	parse_generate_test(&parse_stream,&generate_stream,resource_name,pool);

#if 0
	generate_stream.buffer = generate_buffer;
	generate_stream.pos = generate_buffer;
	generate_stream.size = TEST_BUFFER_SIZE;
	
	generate_parse_test(&generate_stream,pool);
#endif

	fclose(f);
	apr_pool_destroy(pool);
	apr_terminate();
	return 0;
}
