/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include "rtsp_connection.h"
#include "apt_task.h"
#include "apt_log.h"

#define RTSP_MESSAGE_MAX_SIZE 2048

typedef struct rtsp_client_connection_t rtsp_client_connection_t;

struct rtsp_client_connection_t {
	rtsp_connection_t base;

	apr_sockaddr_t   *sockaddr;
	apr_socket_t     *sock; /* connected socket */
};

typedef struct rtsp_client_agent_t rtsp_client_agent_t;

struct rtsp_client_agent_t {
	rtsp_connection_agent_t   base;

	apt_task_t               *task;

	char                     *server_ip;
	apr_port_t                server_port;
	char                     *resource_location;
	rtsp_client_connection_t *connection;

	apr_pool_t               *pool;
};


static apt_bool_t rtsp_agent_destroy(rtsp_connection_agent_t *agent);
static apt_bool_t rtsp_agent_open(rtsp_connection_agent_t *agent);
static apt_bool_t rtsp_agent_close(rtsp_connection_agent_t *agent);
static rtsp_connection_t* rtsp_agent_connect(rtsp_connection_agent_t *agent);
static apt_bool_t rtsp_agent_disconnect(rtsp_connection_t *connection);
static apt_bool_t rtsp_agent_send(rtsp_connection_t *connection, rtsp_message_t *message);
static apt_bool_t rtsp_agent_connection_destroy(rtsp_connection_t *connection);

static const rtsp_connection_agent_method_set_t rtsp_client_agent_method_set = {
	rtsp_agent_destroy,
	rtsp_agent_open,
	rtsp_agent_close,
	rtsp_agent_connect,
	rtsp_agent_disconnect,
	rtsp_agent_send,
	rtsp_agent_connection_destroy
};

static rtsp_client_connection_t *rtsp_agent_connection_create(rtsp_client_agent_t *agent);

static void rtsp_agent_on_terminate_request(void *data);
static void rtsp_agent_main_loop(void *data);


/** Create RTSP client connection agent. */
rtsp_connection_agent_t* rtsp_client_agent_create(rtsp_connection_event_handler_t *event_handler, 
												  const char *server_ip, 
												  apr_port_t server_port, 
												  const char *resource_location,
												  apr_pool_t *pool)
{
	rtsp_client_agent_t *agent;

	if(!server_ip) {
		return NULL;
	}

	apt_log(APT_PRIO_NOTICE,"Create RTSP Connection Agent %s:%hu\n",server_ip,server_port);
	agent = apr_palloc(pool,sizeof(rtsp_client_agent_t));
	agent->pool = pool;
	agent->connection = NULL;

	agent->server_ip = apr_pstrdup(pool,server_ip);
	agent->server_port = server_port;

	agent->resource_location = apr_pstrdup(pool,resource_location);

	agent->task = apt_task_create(agent,rtsp_agent_main_loop,pool);
	if(agent->task) {
		apt_task_event_handler_set(agent->task,TASK_STATE_TERMINATE_REQUESTED,agent,rtsp_agent_on_terminate_request);
	}
	
	agent->base.method_set = &rtsp_client_agent_method_set;
	agent->base.event_handler = event_handler;
	return &agent->base;
}

static apt_bool_t rtsp_agent_destroy(rtsp_connection_agent_t *agent)
{
	rtsp_client_agent_t *client_agent = (rtsp_client_agent_t*)agent;
	apt_log(APT_PRIO_NOTICE,"Destroy RTSP Connection Agent\n");
	if(client_agent->task) {
		apt_task_destroy(client_agent->task);
	}

	return TRUE;
}

static apt_bool_t rtsp_agent_open(rtsp_connection_agent_t *agent)
{
	apt_log(APT_PRIO_INFO,"Open RTSP Connection Agent\n");
	return TRUE;
}

static apt_bool_t rtsp_agent_close(rtsp_connection_agent_t *agent)
{
	rtsp_client_agent_t *client_agent = (rtsp_client_agent_t*)agent;
	apt_log(APT_PRIO_INFO,"Close RTSP Connection Agent\n");
	if(client_agent->connection) {
		rtsp_agent_disconnect(&client_agent->connection->base);
	}
	return TRUE;
}

static rtsp_connection_t* rtsp_agent_connect(rtsp_connection_agent_t *agent)
{
	rtsp_client_agent_t *client_agent = (rtsp_client_agent_t*)agent;
	if(client_agent->connection) {
		return &client_agent->connection->base;
	}

	client_agent->connection = rtsp_agent_connection_create(client_agent);
	if(!client_agent->connection) {
		apt_log(APT_PRIO_WARNING,"Failed to Connect to RTSP Server\n");
		return NULL;
	}
	apt_task_start(client_agent->task);
	return &client_agent->connection->base;
}

static apt_bool_t rtsp_agent_disconnect(rtsp_connection_t *connection)
{
	rtsp_client_agent_t *client_agent = (rtsp_client_agent_t*)connection->agent;
	
	apt_task_terminate(client_agent->task,TRUE);
	client_agent->connection = NULL;
	return TRUE;
}

static apt_bool_t rtsp_agent_url_generate(rtsp_client_agent_t *client_agent, rtsp_message_t *message)
{
	if(message->start_line.message_type == RTSP_MESSAGE_TYPE_REQUEST) {
		if(!message->start_line.common.request_line.resource_name) {
			return FALSE;
		}
		if(client_agent->resource_location && client_agent->resource_location[0] != '\0') {
			message->start_line.common.request_line.url = apr_psprintf(message->pool,
				"rtsp://%s:%d/%s/%s",
				client_agent->server_ip,
				client_agent->server_port,
				client_agent->resource_location,
				message->start_line.common.request_line.resource_name);
		}
		else {
			message->start_line.common.request_line.url = apr_psprintf(message->pool,
				"rtsp://%s:%d/%s",
				client_agent->server_ip,
				client_agent->server_port,
				message->start_line.common.request_line.resource_name);
		}
	}
	return TRUE;
}

static apt_bool_t rtsp_agent_send(rtsp_connection_t *connection, rtsp_message_t *message)
{
	char buffer[RTSP_MESSAGE_MAX_SIZE];
	apt_text_stream_t text_stream;
	size_t size = sizeof(buffer)-1;

	rtsp_client_connection_t *client_connection;
	rtsp_client_agent_t *client_agent;

	if(!connection) {
		return FALSE;
	}
	client_connection = (rtsp_client_connection_t*)connection;
	client_agent = (rtsp_client_agent_t*)connection->agent;
	
	if(!client_connection->sock) {
		return FALSE;
	}

	text_stream.buffer = buffer;
	text_stream.size = size;
	text_stream.pos = text_stream.buffer;

	rtsp_agent_url_generate(client_agent,message);
	if(rtsp_message_generate(message,&text_stream) != TRUE) {
		apt_log(APT_PRIO_WARNING,"Failed to Generate RTSP Message\n");
		return FALSE;
	}
	*text_stream.pos = '\0';
	apt_log(APT_PRIO_DEBUG,"Send RTSP Message size=%lu\n%s\n",text_stream.size,text_stream.buffer);
	apr_socket_send(client_connection->sock,text_stream.buffer,&text_stream.size);
	return TRUE;
}

static apt_bool_t rtsp_agent_connection_destroy(rtsp_connection_t *connection)
{
	rtsp_client_connection_t *client_connection;
	if(!connection || !connection->pool) {
		return FALSE;
	}

	client_connection = (rtsp_client_connection_t*)connection;
	if(client_connection->sock) {
		apr_socket_close(client_connection->sock);
	}

	apr_pool_destroy(connection->pool);
	connection->pool = NULL;
	return TRUE;
}

static rtsp_client_connection_t* rtsp_agent_connection_create(rtsp_client_agent_t *agent)
{
	rtsp_client_connection_t *client_connection;
	apr_status_t status;
	apr_pool_t *pool = NULL;
	if(apr_pool_create(&pool,NULL) != APR_SUCCESS) {
		return NULL;
	}

	client_connection = apr_palloc(pool,sizeof(rtsp_client_connection_t));
	client_connection->base.pool = pool;
	client_connection->base.agent = &agent->base;
	client_connection->base.ref_count = 0;

	client_connection->sockaddr = NULL;
	apr_sockaddr_info_get(&client_connection->sockaddr,agent->server_ip,APR_INET,agent->server_port,0,pool);
	if(!client_connection->sockaddr) {
		apr_pool_destroy(pool);
		return NULL;
	}

	status = apr_socket_create(&client_connection->sock,client_connection->sockaddr->family,SOCK_STREAM,APR_PROTO_TCP,pool);
	if(status != APR_SUCCESS) {
		apr_pool_destroy(pool);
		return NULL;
	}

	apr_socket_opt_set(client_connection->sock, APR_SO_NONBLOCK, 0);
	apr_socket_timeout_set(client_connection->sock, -1);
	apr_socket_opt_set(client_connection->sock, APR_SO_REUSEADDR, 1);

	if(apr_socket_connect(client_connection->sock,client_connection->sockaddr) != APR_SUCCESS) {
		apr_socket_close(client_connection->sock);
		apr_pool_destroy(pool);
		return NULL;
	}

	return client_connection;
}

static apt_bool_t rtsp_agent_messsage_receive(rtsp_client_agent_t *agent)
{
	char buffer[RTSP_MESSAGE_MAX_SIZE];
	apr_size_t size = sizeof(buffer)-1;
	int more_messages_on_buffer = FALSE;
	apr_status_t status;
	apt_text_stream_t text_stream;
	rtsp_message_t *message;
	rtsp_client_connection_t *client_connection = agent->connection;

	if(!client_connection || !client_connection->sock) {
		return FALSE;
	}

	size = sizeof(buffer)-1;
	status = apr_socket_recv(client_connection->sock,buffer,&size);
	if(status == APR_EOF || size == 0) {
		apt_log(APT_PRIO_NOTICE,"Disconnected from RTSP Server\n");
		if(agent->base.event_handler) {
			agent->base.event_handler->event_set->on_disconnect(&client_connection->base);
		}
		agent->connection = NULL;
		return FALSE;
	}

	buffer[size] = '\0';

	text_stream.buffer = buffer;
	text_stream.size = size;
	text_stream.pos = buffer;

	apt_log(APT_PRIO_DEBUG,"Receive RTSP Message size=%lu\n%s\n",text_stream.size,text_stream.buffer);
	do {
		message = rtsp_message_create(RTSP_MESSAGE_TYPE_UNKNOWN,client_connection->base.pool);
		if(rtsp_message_parse(message,&text_stream) == TRUE) {
			if(agent->base.event_handler) {
				apt_log(APT_PRIO_DEBUG,"Signal RTSP Message\n");
				agent->base.event_handler->event_set->on_receive(&client_connection->base,message);
			}
		}
		else {
			apt_log(APT_PRIO_WARNING,"Failed to Parse RTSP Message\n");
			if(message->start_line.message_type == RTSP_MESSAGE_TYPE_REQUEST) {
				rtsp_message_t *response = rtsp_response_create(message,
					RTSP_STATUS_CODE_BAD_REQUEST,RTSP_REASON_PHRASE_BAD_REQUEST,message->pool);
				rtsp_agent_send(&client_connection->base,response);
			}
			
			/* skip to the end of the buffer (support incomplete) */
			text_stream.pos = text_stream.buffer + text_stream.size;
		}

		more_messages_on_buffer = FALSE;
		if(text_stream.size > (size_t)(text_stream.pos - text_stream.buffer)) {
			/* there are more RTSP messages to signal */
			more_messages_on_buffer = TRUE;
			text_stream.size -= text_stream.pos - text_stream.buffer;
			text_stream.buffer = text_stream.pos;
			apt_log(APT_PRIO_DEBUG,"Saving remaining buffer for next message...\n");
		}
	}
	while(more_messages_on_buffer);

	return TRUE;
}

static void rtsp_agent_main_loop(void *data)
{
	rtsp_client_agent_t *agent = data;

	if(!agent || !agent->connection) {
		return;
	}

	apt_log(APT_PRIO_NOTICE,"Connected to RTSP Server\n");
	if(agent->base.event_handler) {
		agent->base.event_handler->event_set->on_connect(&agent->connection->base);
	}

	while(apt_task_state_get(agent->task) != TASK_STATE_TERMINATE_REQUESTED) {
		if(rtsp_agent_messsage_receive(agent) != TRUE) {
			break;
		}
	}
}

static void rtsp_agent_on_terminate_request(void *data)
{
	rtsp_client_agent_t *agent = data;

	if(agent && agent->connection && agent->connection->sock) {
		apr_socket_shutdown(agent->connection->sock,APR_SHUTDOWN_READWRITE);
	}
}
