/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __RTSP_CONNECTION_H__
#define __RTSP_CONNECTION_H__

/**
 * @file rtsp_connection.h
 * @brief RTSP Connection Interface.
 */ 

#include "rtsp_message.h"

APT_BEGIN_EXTERN_C

typedef struct rtsp_connection_t rtsp_connection_t;
typedef struct rtsp_connection_agent_t rtsp_connection_agent_t;
typedef struct rtsp_connection_event_handler_t rtsp_connection_event_handler_t;

typedef struct rtsp_connection_agent_method_set_t rtsp_connection_agent_method_set_t;
/** Set of connection agent virtual methods */
struct rtsp_connection_agent_method_set_t {
	apt_bool_t (*destroy)(rtsp_connection_agent_t *agent);

	apt_bool_t (*open)(rtsp_connection_agent_t *agent);
	apt_bool_t (*close)(rtsp_connection_agent_t *agent);

	rtsp_connection_t* (*connect)(rtsp_connection_agent_t *agent);
	apt_bool_t (*disconnect)(rtsp_connection_t *connection);
	apt_bool_t (*send)(rtsp_connection_t *connection, rtsp_message_t *message);
	apt_bool_t (*destroy_connection)(rtsp_connection_t *connection);
};

typedef struct rtsp_connection_agent_event_set_t rtsp_connection_agent_event_set_t;
/** Set of connection agent events */
struct rtsp_connection_agent_event_set_t {
	apt_bool_t (*on_open)(rtsp_connection_agent_t *agent);
	apt_bool_t (*on_close)(rtsp_connection_agent_t *agent);

	apt_bool_t (*on_connect)(rtsp_connection_t *connection);
	apt_bool_t (*on_disconnect)(rtsp_connection_t *connection);

	apt_bool_t (*on_receive)(rtsp_connection_t *connection, rtsp_message_t *message);
};

/** RTSP connection agent */
struct rtsp_connection_agent_t {
	const rtsp_connection_agent_method_set_t *method_set;

	rtsp_connection_event_handler_t          *event_handler;
};

/** RTSP connection agent event handler */
struct rtsp_connection_event_handler_t {
	const rtsp_connection_agent_event_set_t *event_set;

	void                                    *object;
};

struct rtsp_connection_t {
	rtsp_connection_agent_t *agent;
	apr_pool_t              *pool;
	size_t                   ref_count;
};


/** 
 * Create RTSP client connection agent.
 */
rtsp_connection_agent_t* rtsp_client_agent_create(rtsp_connection_event_handler_t *event_handler, 
												  const char *server_ip, 
												  apr_port_t server_port, 
												  const char *resource_location,
												  apr_pool_t *pool);

/** 
 * Create RTSP server connection agent.
 */
rtsp_connection_agent_t* rtsp_server_agent_create(rtsp_connection_event_handler_t *event_handler, 
												  const char *listen_ip, 
												  apr_port_t listen_port, 
												  const char *resource_location,
												  apr_pool_t *pool);


APT_END_EXTERN_C

#endif /*__RTSP_CONNECTION_H__*/
