/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_GENERIC_HEADER_H__
#define __MRCP_GENERIC_HEADER_H__

/**
 * @file mrcp_generic_header.h
 * @brief MRCP Generic Header
 */ 

#include "mrcp_header_base.h"

MRCP_BEGIN_EXTERN_C

/** Enumaration of MRCP generic headers */
typedef enum {
	GENERIC_HEADER_ACTIVE_REQUEST_ID_LIST,
	GENERIC_HEADER_PROXY_SYNC_ID,
	GENERIC_HEADER_ACCEPT_CHARSET,
	GENERIC_HEADER_CONTENT_TYPE,
	GENERIC_HEADER_CONTENT_ID,
	GENERIC_HEADER_CONTENT_BASE,
	GENERIC_HEADER_CONTENT_ENCODING,
	GENERIC_HEADER_CONTENT_LOCATION,
	GENERIC_HEADER_CONTENT_LENGTH,
	GENERIC_HEADER_CACHE_CONTROL,
	GENERIC_HEADER_LOGGING_TAG,

	GENERIC_HEADER_COUNT
} mrcp_generic_header_id;


#define MAX_ACTIVE_REQUEST_ID_COUNT 5

typedef struct mrcp_request_id_list_t mrcp_request_id_list_t;
/** List (array) of MRCP request identifiers */
struct mrcp_request_id_list_t {
	mrcp_request_id ids[MAX_ACTIVE_REQUEST_ID_COUNT];
	size_t          count;
};


typedef struct mrcp_generic_header_t mrcp_generic_header_t;

/** MRCP generic header */
struct mrcp_generic_header_t {
	/** Indicates the list of request-ids to which it should apply */
	mrcp_request_id_list_t active_request_id_list;
	/** Helps the resource receiving the event, proxied by the client, 
	to decide if this event has been processed through a direct interaction of the resources */
	char                   *proxy_sync_id;
	/** Specifies the acceptable character set for entities returned in the response or events associated with this request */
	char                   *accept_charset;
	/** Restricted to speech markup, grammar, recognition results, etc. */
	char                   *content_type;
	/** Contains an ID or name for the content, by which it can be referred to */
	char                   *content_id;
	/** May be used to specify the base URI for resolving relative URLs within the entity */
	char                   *content_base;
	/** Indicates what additional content coding has been applied to the entity-body */
	char                   *content_encoding;
	/** Statement of the location of the resource corresponding to this particular entity at the time of the request */
	char                   *content_location;
	/** Contains the length of the content of the message body */
	size_t                  content_length;
	/** Defines the default caching algorithms on the media server for the session or request */
	char                   *cache_control;
	/** Set the logging tag for logs generated by the media server */
	char                   *logging_tag;
};


/** Initialize generic header */
void mrcp_generic_header_init(mrcp_generic_header_t *generic_header);

/** Destroy generic header */
void mrcp_generic_header_destroy(mrcp_generic_header_t *generic_header);


/** Initialize generic header accessor */
void mrcp_generic_header_accessor_init(mrcp_header_accessor_t *header_accessor);

/** Destroy generic header accessor */
void mrcp_generic_header_accessor_destroy(mrcp_header_accessor_t *header_accessor);



/** Append generic header active request id list */
static APR_INLINE mrcp_status_t active_request_id_list_append(mrcp_generic_header_t *generic_header, mrcp_request_id request_id)
{
	if(generic_header->active_request_id_list.count >= MAX_ACTIVE_REQUEST_ID_COUNT) {
		return MRCP_STATUS_FAILURE;
	}
	generic_header->active_request_id_list.ids[generic_header->active_request_id_list.count++] = request_id;
	return MRCP_STATUS_SUCCESS;
}

MRCP_END_EXTERN_C

#endif /*__MRCP_GENERIC_HEADER_H__*/
