/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_MODULE_H__
#define __MRCP_MODULE_H__

/**
 * @file mrcp_module.h
 * @brief MRCP Module Definition.
 */ 

#include "mrcp_types.h"
#include "apt_task_msg.h"

MRCP_BEGIN_EXTERN_C

/** Module state enum */
typedef enum {
	MODULE_STATE_NONE,
	MODULE_STATE_OPEN_INPROGRESS,
	MODULE_STATE_OPENED,
	MODULE_STATE_CLOSE_INPROGRESS,
	MODULE_STATE_CLOSED,
} mrcp_module_state_t;

typedef struct mrcp_module_t mrcp_module_t;
typedef struct mrcp_engine_t mrcp_engine_t;

typedef struct mrcp_module_method_set_t mrcp_module_method_set_t;
/** Set of module virtual methods */
struct mrcp_module_method_set_t {
	mrcp_status_t (*destroy)(mrcp_module_t *module);

	mrcp_module_state_t (*open)(mrcp_module_t *module);
	mrcp_module_state_t (*close)(mrcp_module_t *module);

	mrcp_status_t (*process)(mrcp_module_t *module, apt_task_msg_t *msg);
};

typedef struct mrcp_module_event_set_t mrcp_module_event_set_t;
/** Set of module virtual events */
struct mrcp_module_event_set_t {
	mrcp_status_t (*on_open)(mrcp_module_t *module);
	mrcp_status_t (*on_close)(mrcp_module_t *module);
};

/** Module is an abstract base of agents and plugins */
struct mrcp_module_t {
	const mrcp_module_method_set_t *method_set;
	const mrcp_module_event_set_t  *event_set;

	mrcp_module_state_t             state;

	mrcp_engine_t                  *engine;
	int                             type;

	mrcp_status_t (*signal)(mrcp_module_t *module, apt_task_msg_t *task_msg);
};

/** Initialize MRCP module */
static APR_INLINE void mrcp_module_init(mrcp_module_t *module, const mrcp_module_method_set_t *method_set)
{
	module->method_set = method_set;
	module->event_set = NULL;
	module->state = MODULE_STATE_NONE;
	module->engine = NULL;
	module->type = 0;
	module->signal = NULL;
}

/** Validate MRCP module */
static APR_INLINE mrcp_status_t mrcp_module_validity_check(mrcp_module_t *module)
{
	if(module->method_set && module->method_set->open && module->method_set->close && 
		module->method_set->destroy && module->method_set->process) {
		return MRCP_STATUS_SUCCESS;
	}
	return MRCP_STATUS_FAILURE;
}

MRCP_END_EXTERN_C

#endif /*__MRCP_MODULE_H__*/
