/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_MEDIA_AGENT_H__
#define __MRCP_MEDIA_AGENT_H__

/**
 * @file mrcp_media_agent.h
 * @brief MRCP Media Agent Definition.
 */ 

#include "mrcp_types.h"
#include "mrcp_module.h"
#include "mrcp_audio_channel.h"

MRCP_BEGIN_EXTERN_C

typedef struct mrcp_media_agent_method_set_t mrcp_media_agent_method_set_t;
typedef struct mrcp_media_agent_event_set_t mrcp_media_agent_event_set_t;

/** Media agent base */
struct mrcp_media_agent_t {
	/** Media agent is an MRCP module */
	mrcp_module_t                        module;

	/** Set of signaling agent virtual methods */
	const mrcp_media_agent_method_set_t *method_set;
	/** Actual data set by inherited object */
	void                                *object;

	/** Agent state */
	int                                  state;

	/** Local IP address */
	char                                *ip;
	/** RTP port range (min) */
	unsigned short                       rtp_port_min;
	/** RTP port range (max) */
	unsigned short                       rtp_port_max;
	/** RTP port range (cur) */
	unsigned short                       rtp_port_cur;
};


/** Set of media agent virtual methods */
struct mrcp_media_agent_method_set_t {
	/** Create new audio channel */
	mrcp_status_t (*channel_create)(
								mrcp_media_agent_t *agent,
								mrcp_audio_channel_t *channel,
								apr_pool_t *pool);
};


/**
 * Create media agent object.
 * @param method_set the implemented set of signaling agent methods
 * @param object the associated (inherited) object
 * @param pool the memory pool to allocate memory from
 * @return the signaling agent base object
 */
mrcp_media_agent_t* mrcp_media_agent_create(
						const mrcp_media_agent_method_set_t *media_method_set, 
						const mrcp_module_method_set_t *module_method_set, 
						void *object,
						apr_pool_t *pool);

/**
 * Retrieve the object associated with media agent upon creation.
 * @param agent the agent to retrieve from
 * @return the object
 */
void* mrcp_media_agent_object_get(mrcp_media_agent_t *agent);

/**
 * Signal task message.
 * @param agent the agent to signal message from
 * @param task_msg the message to signal
 */
mrcp_status_t mrcp_media_agent_signal(mrcp_media_agent_t *agent, apt_task_msg_t *task_msg);

/** Perform validity check for specified agent */
static APR_INLINE mrcp_status_t mrcp_media_agent_validity_check(mrcp_media_agent_t *agent)
{
	return mrcp_module_validity_check(&agent->module);
}

/**
 * Create audio channel object.
 * @param agent the media agent (creator of the channel)
 * @param method_set the implemented set of audio channel methods
 * @param object the associated (inherited) object
 * @param pool the memory pool to allocate memory from
 * @return the audio channel base object
 */
mrcp_audio_channel_t* mrcp_audio_channel_create(
									mrcp_audio_channel_mode_t mode,
									mrcp_media_audio_t *local_media,
									mrcp_media_audio_t *remote_media,
									apr_pool_t *pool);

/**
 * Send media request.
 * @param agent the media agent (creator of the channel)
 * @param media the local media to process
 * @param pool the memory pool to allocate memory from
 */
mrcp_status_t mrcp_media_agent_request(
									mrcp_media_agent_t *agent,
									mrcp_media_audio_t *media,
									apr_pool_t *pool);



MRCP_END_EXTERN_C

#endif /*__MRCP_MEDIA_AGENT_H__*/
