/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MRCP_CLIENT_CONTEXT_H__
#define __MRCP_CLIENT_CONTEXT_H__

/**
 * @file mrcp_client_context.h
 * @brief MRCP Client Context (interface granted to user level application).
 */ 

#include "mrcp_client_types.h"
#include "mrcp_message.h"
#include "mrcp_description.h"
#include "audio_source.h"
#include "audio_sink.h"

MRCP_BEGIN_EXTERN_C


/** 
 * Create client context instance.
 * @param object the external object to associate with created context
 * @param event_handler the handler of MRCP client events to associate with created context
 * @return the created client context instance
 */
mrcp_client_context_t* mrcp_client_context_create(void *object, mrcp_client_event_handler_t *event_handler);

/** 
 * Destroy client context instance.
 * @param context the context to destroy
 */
mrcp_status_t mrcp_client_context_destroy(mrcp_client_context_t *context);


/** 
 * Create client session.
 * @param context the entire context
 * @param object the external object to associate with created session
 * @return the created session
 */
mrcp_session_t* mrcp_client_context_session_create(mrcp_client_context_t *context, void *object);

/** 
 * Destroy client session (Session must be terminated prior to destroy).
 * @param context the entire context
 * @param session the session to destroy
 */
mrcp_status_t mrcp_client_context_session_destroy(mrcp_client_context_t *context, mrcp_session_t *session);

/** 
 * Send session initiation request (optional).
 * @param context the entire context
 * @param session the session to initiate
 */
mrcp_status_t mrcp_client_context_session_initiate(mrcp_client_context_t *context, mrcp_session_t *session);

/** 
 * Send session termination request.
 * @param context the entire context
 * @param session the session to terminate
 */
mrcp_status_t mrcp_client_context_session_terminate(mrcp_client_context_t *context, mrcp_session_t *session);

/** 
 * Create synthesizer channel and send channel addition request.
 * @param context the entire context
 * @param session the session to create channel in
 * @param sink the audio sink to receive synthesized audio data
 * @return the created channel
 */
mrcp_client_channel_t* mrcp_client_synthesizer_channel_create(mrcp_client_context_t *context, mrcp_session_t *session, audio_sink_t *sink);

/** 
 * Create recognizer channel and send channel addition request.
 * @param context the entire context
 * @param session the session to create channel in
 * @param source the source of audio data to transmit
 * @return the created channel
 */
mrcp_client_channel_t* mrcp_client_recognizer_channel_create(mrcp_client_context_t *context, mrcp_session_t *session, audio_source_t *source);

/** 
 * Send channel destruction request.
 * @param context the entire context
 * @param session the session to destroy channel from
 * @param channel the channel to destroy
 */
mrcp_status_t mrcp_client_context_channel_destroy(mrcp_client_context_t *context, mrcp_session_t *session, mrcp_client_channel_t *channel);

/** 
 * Send channel creation (addition) request.
 * @param context the entire context
 * @param session the session to add channel to
 * @param control_channel the MRCP control channel to add
 * @param audio_channel the associated RTP audio channel
 */
mrcp_status_t mrcp_client_context_channel_add(mrcp_client_context_t *context, mrcp_session_t *session, mrcp_client_channel_t *control_channel, mrcp_audio_channel_t *audio_channel);

/** 
 * Send channel destruction (removal) request.
 * @param context the entire context
 * @param session the session to remove channel from
 * @param channel the channel to remove
 */
mrcp_status_t mrcp_client_context_channel_remove(mrcp_client_context_t *context, mrcp_session_t *session, mrcp_client_channel_t *control_channel);

/** 
 * Create RTP audio attributes.
 * @param context the entire context
 * @param session the session to create media in
 * @return the created media
 */
mrcp_media_audio_t* mrcp_client_audio_media_create(mrcp_client_context_t *context, mrcp_session_t *session, const char *ip, unsigned short port);

/** 
 * Create RTP audio channel.
 * @param context the entire context
 * @param session the session to create channel in
 * @param local_media the media attributes (IP, port, codecs)
 * @param mode the mode of the channel (passive, active, none)
 * @return the created channel
 */
mrcp_audio_channel_t* mrcp_client_audio_channel_create(mrcp_client_context_t *context, mrcp_session_t *session, mrcp_media_audio_t *local_media, mrcp_audio_channel_mode_t mode);

/** 
 * Retrieve the audio sink of the channel.
 * @param channel the channel to get sink from
 */
audio_sink_t* mrcp_client_audio_sink_get(mrcp_audio_channel_t *channel);

/** 
 * Retrieve the audio source of the channel.
 * @param channel the channel to get source from
 */
audio_source_t* mrcp_client_audio_source_get(mrcp_audio_channel_t *channel);

/** 
 * Send MRCP message to modify channel.
 * @param context the entire context
 * @param session the session channel belongs to
 * @param channel the channel to remove
 * @param mrcp_message the message to send
 */
mrcp_status_t mrcp_client_context_channel_modify(mrcp_client_context_t *context, mrcp_session_t *session, mrcp_message_t *mrcp_message);

/** 
 * Get MRCP message prepared to modify channel.
 * @param context the entire context
 * @param session the session channel belongs to
 * @param channel the channel MRCP message will be created (prepared) for
 * @param method_id the method identifier
 * @return the prepared MRCP message
 */
mrcp_message_t* mrcp_client_context_message_get(mrcp_client_context_t *context, mrcp_session_t *session, mrcp_client_channel_t *channel, mrcp_method_id method_id);

/** 
 * Retrieve the external object associated with entire context upon creation.
 * @param context the entire context
 * @return the retrieved external object
 */
void* mrcp_client_context_object_get(mrcp_client_context_t *context);

/** 
 * Retrieve the external object associated with entire session upon creation.
 * @param session the entire session
 * @return the retrieved external object
 */
void* mrcp_client_context_session_object_get(mrcp_session_t *session);

/** 
 * Send resource discovery request.
 * @param context the entire context
 */
mrcp_status_t mrcp_client_context_resource_discover(mrcp_client_context_t *context);

/** 
 * Get MRCP resource identifier of entire channel.
 * @param channel the channel to get identifier from
 * @return the resource identifier
 */
mrcp_resource_id mrcp_client_context_resource_id_get(mrcp_client_channel_t *channel);


/** Set of client event handlers */
struct mrcp_client_event_handler_t {
	mrcp_status_t (*on_session_initiate)(mrcp_client_context_t *context, mrcp_session_t *session);

	mrcp_status_t (*on_session_terminate)(mrcp_client_context_t *context, mrcp_session_t *session);

	mrcp_status_t (*on_channel_add)(mrcp_client_context_t *context, mrcp_session_t *session, mrcp_client_channel_t *control_channel, mrcp_audio_channel_t *audio_channel);

	mrcp_status_t (*on_channel_remove)(mrcp_client_context_t *context, mrcp_session_t *session, mrcp_client_channel_t *control_channel);

	mrcp_status_t (*on_channel_modify)(mrcp_client_context_t *context, mrcp_session_t *session, mrcp_message_t *mrcp_message);

	mrcp_status_t (*on_resource_discover)(mrcp_client_context_t *context, mrcp_descriptor_t *capabilities);
};


MRCP_END_EXTERN_C

#endif /*__MRCP_CLIENT_CONTEXT_H__*/
