/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __MEDIA_PROCESSOR_H__
#define __MEDIA_PROCESSOR_H__

/**
 * @file media_processor.h
 * @brief Media Processor
 */ 

#include "media_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/** Media request types */
typedef enum {
	MEDIA_REQUEST_TYPE_SYNC, /* wait for request to be processed */
	MEDIA_REQUEST_TYPE_ASYNC /* no wait, async call */
} media_request_type;


/** Create media processor */
media_processor_t *media_processor_create(apr_pool_t *pool);

/** Start media processor */
apr_status_t media_processor_start(media_processor_t *processor);

/** Terminate media processor */
apr_status_t media_processor_terminate(media_processor_t *processor);

/** Destroy media processor */
apr_status_t media_processor_destroy(media_processor_t *processor);

/** Add media context to media processor */
apr_status_t media_processor_context_add(media_processor_t *processor, media_context_t *context, media_request_type type);

/** Remove media context from media processor */
apr_status_t media_processor_context_remove(media_processor_t *processor, media_context_t *context, media_request_type type);

/** Add audio source to media context */
apr_status_t media_context_source_add(media_processor_t *processor, media_context_t *context, audio_source_t *source, media_request_type type);

/** Remove audio source from media context */
apr_status_t media_context_source_remove(media_processor_t *processor, media_context_t *context, audio_source_t *source, media_request_type type);

/** Add audio sink to media context */
apr_status_t media_context_sink_add(media_processor_t *processor, media_context_t *context, audio_sink_t *sink, media_request_type type);

/** Remove audio sink from media context */
apr_status_t media_context_sink_remove(media_processor_t *processor, media_context_t *context, audio_sink_t *sink, media_request_type type);


/** Create media connector (type of media context) */
media_context_t* media_connector_create(apr_uint16_t sampling_rate, apr_pool_t *pool);

/** Create media mixer (type of media context) */
media_context_t* media_mixer_create(apr_uint16_t sampling_rate, apr_pool_t *pool);

/** Create media multiplexor (type of media context) */
media_context_t* media_multiplexor_create(apr_uint16_t sampling_rate, apr_pool_t *pool);

/** Remove media context */
apr_status_t media_context_destroy(media_context_t *context);


/** Get codec manager from media processor */
codec_manager_t* media_processor_codec_manager_get(media_processor_t *processor);

#ifdef __cplusplus
}
#endif

#endif /*__MEDIA_PROCESSOR_H__*/
