/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __APT_TASK_H__
#define __APT_TASK_H__

/**
 * @file apt_task.h
 * @brief Thread Execution Abstraction
 */ 

#include "apt.h"

APT_BEGIN_EXTERN_C

/** Enumaration of task states */
typedef enum {
	TASK_STATE_START_REQUESTED,
	TASK_STATE_START_IN_PROGRESS,
	TASK_STATE_START_COMPLETED,

	TASK_STATE_TERMINATE_REQUESTED,
	TASK_STATE_TERMINATE_IN_PROGRESS,
	TASK_STATE_TERMINATE_COMPLETED,

	TASK_STATE_COUNT,
	TASK_STATE_NONE = TASK_STATE_COUNT
} apt_task_state_t;


/** Prototype of task event handler */
typedef void (*apt_task_event_handler)(void *data);


/** Opaque task declaration */
typedef struct apt_task_t apt_task_t;

/**
 * Create task.
 * @param data the data to pass to main function of the task
 * @param main the the main function of the task to run
 * @param pool the pool to allocate memory from
 */
APT_DECLARE(apt_task_t*) apt_task_create(void *data, apt_task_event_handler main, apr_pool_t *pool);

/**
 * Destroy task.
 * @param task the task to destroy
 */
APT_DECLARE(apt_bool_t) apt_task_destroy(apt_task_t *task);

/**
 * Start task.
 * @param task the task to start
 */
APT_DECLARE(apt_bool_t) apt_task_start(apt_task_t *task);

/**
 * Terminate task.
 * @param task the task to terminate
 * @param wait_until_terminate
 */
APT_DECLARE(apt_bool_t) apt_task_terminate(apt_task_t *task, apt_bool_t wait_until_terminate);

/**
 * Wait for task to terminate.
 * @param task the task to wait
 */
APT_DECLARE(apt_bool_t) apt_task_wait_until_terminate(apt_task_t *task);

/**
 * Hold task execution.
 * @param task the task to hold
 * @param msec the time to hold
 */
APT_DECLARE(void) apt_task_delay(apr_size_t msec);

/**
 * Set task event handler.
 * @param task the task to set
 * @param task_event
 * @param data the data to pass to event handler function
 * @param handler the event handler function
 */
APT_DECLARE(apt_bool_t) apt_task_event_handler_set(apt_task_t *task, apt_task_state_t task_event,
                                        void *data, apt_task_event_handler handler);

/**
 * Set task state.
 * @param task the task to set
 * @param state
 */
APT_DECLARE(apt_task_state_t) apt_task_state_set(apt_task_t *task, apt_task_state_t state);

/**
 * Get task state.
 * @param task the task to get state of
 */
APT_DECLARE(apt_task_state_t) apt_task_state_get(apt_task_t *task);

/**
 * Get task data (external data associated with the task).
 * @param task the task to get data from
 */
APT_DECLARE(void*) apt_task_data_get(apt_task_t *task);


APT_END_EXTERN_C

#endif /*__APT_TASK_H__*/
