/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __APT_PRODUCER_TASK_H__
#define __APT_PRODUCER_TASK_H__

/**
 * @file mrcp_producer_task.h
 * @brief Producer Task Definition
 *        Producer task sends messages to consumer task to process.
 */ 

#include "apt_task_msg.h"

APT_BEGIN_EXTERN_C

/** Opaque producer task declaration */
typedef struct apt_producer_task_t apt_producer_task_t;

/**
 * Create producer task.
 * @param data the data to pass to main function of the task
 * @param main the the main function of the task to run
 * @param msg_pool the message pool task messages are allocated from
 * @param pool the pool to allocate consumer task from
 * @return the created producer task
 */
APT_DECLARE(apt_producer_task_t*) apt_producer_task_create(void *data, apt_task_event_handler main, 
                                      apt_task_msg_pool_t *msg_pool, apr_pool_t *pool);

/**
 * Destroy producer task.
 * @param producer_task the producer task to destroy
 */
APT_DECLARE(apt_bool_t) apt_producer_task_destroy(apt_producer_task_t *producer_task);

/**
 * Start producer task.
 * @param producer_task the producer task to start
 * @param task_param the params to pass to producer task
 */
APT_DECLARE(apt_bool_t) apt_producer_task_start(apt_producer_task_t *producer_task);

/**
 * Terminate producer task.
 * @param producer_task the producer task to terminate
 */
APT_DECLARE(apt_bool_t) apt_producer_task_terminate(apt_producer_task_t *producer_task, 
                                                      apt_bool_t wait_until_terminate);

/**
 * Get task message from the message pool associated with producer task.
 * @param producer_task the producer task to get message from
 */
APT_DECLARE(apt_task_msg_t*) apt_producer_task_msg_get(apt_producer_task_t *producer_task);

/**
 * Retrieve task instance.
 * @param producer_task the producer task to get instance from
 */
APT_DECLARE(apt_task_t*) apt_producer_task_get(apt_producer_task_t *producer_task);

/**
 * Indicate whether producer task is terminating.
 * @param producer_task the producer task to query
 */
APT_DECLARE(apt_bool_t) apt_producer_task_is_terminating(apt_producer_task_t *producer_task);

APT_END_EXTERN_C

#endif /*__APT_PRODUCER_TASK_H__*/
