/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#ifndef __APT_LOG_H__
#define __APT_LOG_H__

/**
 * @file aptlog.h
 * @brief Logging Capabilities
 */ 

#include <stdio.h>
#include <stdarg.h>
#include "apt.h"

APT_BEGIN_EXTERN_C

/** Priority of log messages ordered from highest priority to lowest (rfc3164) */
typedef enum {
	/** System is unusable */
	APT_PRIO_EMERGENCY,
	/** Action must be taken immediately */
	APT_PRIO_ALERT,
	/** Critical condition */
	APT_PRIO_CRITICAL,
	/** Error condition */
	APT_PRIO_ERROR,
	/** Warning condition */
	APT_PRIO_WARNING,
	/** Normal, but significant, condition */
	APT_PRIO_NOTICE,
	/** Informational message */
	APT_PRIO_INFO,
	/** Debug-level message */
	APT_PRIO_DEBUG,

	/** Number of priorities */
	APT_PRIO_COUNT
} apt_log_priority_t;

/** Prototype of log handler function */
typedef apt_bool_t (*apt_log_handler)(apt_log_priority_t priority, const char *format, va_list arg_ptr);


/**
 * Initialize the logger.
 * @param priority the priority of logging
 */
APT_DECLARE(apt_bool_t) apt_log_initialize(apt_log_priority_t priority);

/**
 * Set priority of logging.
 * @param priority the priority of logging
 */
APT_DECLARE(void) apt_log_priority_set(apt_log_priority_t priority);

/**
 * Set external log handler.
 * @param handler the handler to pass log events to
 * @remark printf is used if external log handler isn't set
 */
APT_DECLARE(void) apt_log_handler_set(apt_log_handler handler);

/**
 * Do logging.
 * @param priority the priority of the entire log entry
 * @param format the format of the entire log entry
 */
APT_DECLARE(apt_bool_t) apt_log(apt_log_priority_t priority, const char *format, ...);

APT_END_EXTERN_C

#endif /*__APT_LOG_H__*/
