/*
 * OpenMRCP - Open Source Media Resource Control Protocol Stack
 * Copyright (C) 2007, Cepstral LLC
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Author(s):
 * Arsen Chaloyan <achaloyan@yahoo.com>
 *
 * Contributor(s):
 *
 */

#include <assert.h>
#include <apr_hash.h>
#include "apt_consumer_task.h"
#include "apt_unique_id.h"
#include "apt_ptr_queue.h"
#include "apt_log.h"
#include "rtsp_engine.h"
#include "rtsp_connection.h"

#define RTSP_SESSION_ID_HEX_STRING_LENGTH 16

/** RTSP Engine */
struct rtsp_engine_t {
	/** Main task waiting for incoming messages */
	apt_consumer_task_t           *consumer_task;
	
	/** Message pool to allocate task messages from */
	apt_task_msg_pool_t           *msg_pool;
	
	/** Table of RTSP sessions */
	apr_hash_t                     *session_table;

	/** Queue of RTSP requests sent and waiting for the responses */
	apr_hash_t                     *request_queue;
	size_t                          last_cseq;

	rtsp_engine_event_handler_t    *handler;

	rtsp_connection_event_handler_t connection_handler;
	rtsp_connection_agent_t        *connection_agent;

	apr_pool_t                     *pool;
};

/** RTSP Session */
struct rtsp_session_t {
	/** The owner of the session */
	rtsp_engine_t       *engine;
	/** Session identifier */
	rtsp_session_id      id;

	/** Connection used as transport for RTSP messages */
	rtsp_connection_t   *connection;
	/** Queue of pending RTSP requests */
	apt_ptr_queue_t     *queue;

	/** Context specific object */
	void                *object;

	apr_pool_t          *pool;
};

typedef enum {
	RTSP_CONNECTION_MSG_AGENT_OPEN,
	RTSP_CONNECTION_MSG_AGENT_CLOSE,
	RTSP_CONNECTION_MSG_PEER_CONNECT,
	RTSP_CONNECTION_MSG_PEER_DISCONNECT,
	RTSP_CONNECTION_MSG_RECEIVE,

	RTSP_USER_MSG_SESSION_CREATE,
	RTSP_USER_MSG_SESSION_TERMINATE,
	RTSP_USER_MSG_SEND

} rtsp_engine_msg_id;

typedef struct rtsp_engine_msg_t rtsp_engine_msg_t;

struct rtsp_engine_msg_t {
	rtsp_engine_msg_id       id;
	rtsp_connection_agent_t *agent;
	rtsp_connection_t       *connection;
	rtsp_message_t          *message;
	rtsp_session_t          *session;
};

static apt_bool_t rtsp_engine_connection_task_msg_signal(
										rtsp_engine_msg_id       id,
										rtsp_connection_agent_t *agent,
										rtsp_connection_t       *connection,
										rtsp_message_t          *message );

static apt_bool_t rtsp_engine_user_task_msg_signal(
										rtsp_engine_msg_id id,
										rtsp_session_t    *session,
										rtsp_message_t    *message );

static rtsp_session_t* rtsp_session_create(rtsp_engine_t *engine, apt_bool_t generate_session_id);
static apt_bool_t rtsp_session_destroy(rtsp_session_t *session);

static apt_bool_t rtsp_engine_msg_handler(void *object, apt_task_msg_t *task_msg);

static void rtsp_engine_on_start_in_progress(void *data);
static void rtsp_engine_on_terminate_in_progress(void *data);
static void rtsp_engine_on_terminate_complete(void *data);

static apt_bool_t rtsp_engine_on_agent_open(rtsp_connection_agent_t *agent);
static apt_bool_t rtsp_engine_on_agent_close(rtsp_connection_agent_t *agent);
static apt_bool_t rtsp_engine_on_peer_connect(rtsp_connection_t *connection);
static apt_bool_t rtsp_engine_on_peer_disconnect(rtsp_connection_t *connection);
static apt_bool_t rtsp_engine_on_message_receive(rtsp_connection_t *connection, rtsp_message_t *message);

static const rtsp_connection_agent_event_set_t connection_event_set = {
	rtsp_engine_on_agent_open,
	rtsp_engine_on_agent_close,
	rtsp_engine_on_peer_connect,
	rtsp_engine_on_peer_disconnect,
	rtsp_engine_on_message_receive
};


/** Create RTSP engine. */
rtsp_engine_t* rtsp_engine_create(rtsp_engine_event_handler_t *handler)
{
	rtsp_engine_t *engine;
	apt_task_t *task;
	apr_pool_t *pool;

	if(!handler) {
		return NULL;
	}

	if(apr_pool_create(&pool,NULL) != APR_SUCCESS) {
		return NULL;
	}

	engine = apr_palloc(pool,sizeof(rtsp_engine_t));
	engine->pool = pool;
	engine->handler = handler;

	engine->connection_handler.event_set = &connection_event_set;
	engine->connection_handler.object = engine;

	engine->session_table = apr_hash_make(pool);
	engine->request_queue = apr_hash_make(pool);
	engine->last_cseq = 0;

	engine->consumer_task = apt_consumer_task_create(engine,rtsp_engine_msg_handler,pool);
	if(!engine->consumer_task) {
		apt_log(APT_PRIO_DEBUG,"Failed to Create Server Consumer Task\n");
		apr_pool_destroy(pool);
		return NULL;
	}
	
	task = apt_consumer_task_get(engine->consumer_task);
	apt_task_event_handler_set(task,TASK_STATE_START_IN_PROGRESS,engine,rtsp_engine_on_start_in_progress);
	apt_task_event_handler_set(task,TASK_STATE_TERMINATE_IN_PROGRESS,engine,rtsp_engine_on_terminate_in_progress);
	apt_task_event_handler_set(task,TASK_STATE_TERMINATE_COMPLETED,engine,rtsp_engine_on_terminate_complete);
	
	engine->msg_pool = apt_task_msg_pool_create_dynamic(sizeof(rtsp_engine_msg_t),pool);
	
	return engine;
}

/** Start RTSP client engine. */
rtsp_engine_t* rtsp_client_engine_start(rtsp_engine_event_handler_t *handler, const char *server_ip, apr_port_t server_port, const char *resource_location)
{
	rtsp_engine_t *engine = rtsp_engine_create(handler);
	if(!engine) {
		return NULL;
	}

	apt_log(APT_PRIO_NOTICE,"Starting RTSP Client Engine\n");
	engine->connection_agent = rtsp_client_agent_create(&engine->connection_handler,server_ip,server_port,resource_location,engine->pool);
	apt_consumer_task_start(engine->consumer_task);
	return engine;
}

/** Start RTSP server engine. */
rtsp_engine_t* rtsp_server_engine_start(rtsp_engine_event_handler_t *handler, const char *listen_ip, apr_port_t listen_port, const char *resource_location)
{
	rtsp_engine_t *engine = rtsp_engine_create(handler);
	if(!engine) {
		return NULL;
	}

	apt_log(APT_PRIO_NOTICE,"Starting RTSP Server Engine\n");
	engine->connection_agent = rtsp_server_agent_create(&engine->connection_handler,listen_ip,listen_port,resource_location,engine->pool);
	apt_consumer_task_start(engine->consumer_task);
	return engine;
}

/** Shutdown RTSP engine. */
apt_bool_t rtsp_engine_shutdown(rtsp_engine_t *engine)
{
	apr_pool_t *pool;
	if(!engine || !engine->pool) {
		return FALSE;
	}

	apt_log(APT_PRIO_NOTICE,"Shutting Down RTSP Engine\n");
	if(apt_consumer_task_terminate(engine->consumer_task) != TRUE) {
		apt_log(APT_PRIO_DEBUG,"Failed to Terminate Server Consumer Task\n");
		return FALSE;
	}
	
	if(engine->msg_pool) {
		apt_task_msg_pool_destroy(engine->msg_pool);
		engine->msg_pool = NULL;
	}
	apt_consumer_task_destroy(engine->consumer_task);
	engine->consumer_task = NULL;
	engine->session_table = NULL;
	engine->request_queue = NULL;
	
	pool = engine->pool;
	engine->pool = NULL;

	apr_pool_destroy(pool);
	return TRUE;
}

rtsp_session_t* rtsp_engine_session_create(rtsp_engine_t *engine, rtsp_message_t *message)
{
	rtsp_session_t *session = rtsp_session_create(engine,FALSE);

	rtsp_engine_user_task_msg_signal(
										RTSP_USER_MSG_SESSION_CREATE,
										session,
										message);

	return session;
}

apt_bool_t rtsp_engine_message_send(rtsp_session_t *session, rtsp_message_t *message)
{
	return rtsp_engine_user_task_msg_signal(
										RTSP_USER_MSG_SEND,
										session,
										message);
}

apt_bool_t rtsp_engine_session_terminate(rtsp_session_t *session)
{
	return rtsp_engine_user_task_msg_signal(
										RTSP_USER_MSG_SESSION_TERMINATE,
										session,
										NULL);
}

apt_bool_t rtsp_engine_session_destroy(rtsp_session_t *session)
{
	return rtsp_session_destroy(session);
}

void rtsp_engine_session_object_set(rtsp_session_t *session, void *object)
{
	session->object = object;
}

void* rtsp_engine_session_object_get(rtsp_session_t *session)
{
	return session->object;
}

rtsp_session_id* rtsp_engine_session_id_get(rtsp_session_t *session)
{
	return &session->id;
}


static rtsp_session_t* rtsp_session_create(rtsp_engine_t *engine, apt_bool_t generate_session_id)
{
	rtsp_session_t *session;
	apr_pool_t *pool = NULL;
	if(apr_pool_create(&pool,NULL) != APR_SUCCESS) {
		return NULL;
	}
	session = apr_palloc(pool,sizeof(rtsp_session_t));
	session->pool = pool;
	session->engine = engine;
	session->connection = NULL;
	session->id.hex_str = NULL;
	session->id.length = 0;
	session->object = NULL;

	session->queue = apt_ptr_queue_create(pool);

	if(generate_session_id == TRUE) {
		apt_unique_id_generate(&session->id,RTSP_SESSION_ID_HEX_STRING_LENGTH,session->pool);
		apt_log(APT_PRIO_NOTICE,"Create RTSP Session <%s>\n",session->id.hex_str);
	}

	return session;
}

static apt_bool_t rtsp_session_destroy(rtsp_session_t *session)
{
	if(!session || !session->pool) {
		return FALSE;
	}
	if(session->queue) {
		apt_ptr_queue_destroy(session->queue);
	}
	apr_pool_destroy(session->pool);
	session->pool = NULL;
	return TRUE;
}

static apt_bool_t rtsp_session_connection_attach(rtsp_engine_t *engine, rtsp_session_t *session, rtsp_connection_t *connection)
{
	if(!session || !connection) {
		return FALSE;
	}
	/* add ref */
	connection->ref_count++;
	session->connection = connection;
	return TRUE;
}

static apt_bool_t rtsp_session_connection_dettach(rtsp_engine_t *engine, rtsp_session_t *session)
{
	if(!session || !session->connection) {
		return FALSE;
	}
	/* remove ref */
	if(--session->connection->ref_count <= 1) {
		engine->connection_agent->method_set->disconnect(session->connection);
	}
	session->connection = NULL;
	return TRUE;
}


static APR_INLINE rtsp_session_t* rtsp_engine_session_find(rtsp_engine_t *engine, const char *session_id, size_t length)
{
	return apr_hash_get(engine->session_table,session_id,length);
}

static apt_bool_t rtsp_engine_session_table_add(rtsp_engine_t *engine, rtsp_session_t *session)
{
	if(!session || !session->id.hex_str) {
		return FALSE;
	}
	apt_log(APT_PRIO_NOTICE,"Add RTSP Session <%s>\n",session->id.hex_str);
	apr_hash_set(engine->session_table,session->id.hex_str,session->id.length,session);
	return TRUE;
}

static apt_bool_t rtsp_engine_session_table_remove(rtsp_engine_t *engine, rtsp_session_t *session)
{
	if(!session || !session->id.hex_str) {
		return FALSE;
	}
	apt_log(APT_PRIO_NOTICE,"Remove RTSP Session <%s>\n",session->id.hex_str);
	apr_hash_set(engine->session_table,session->id.hex_str,session->id.length,NULL);
	return TRUE;
}

static apt_bool_t rtsp_engine_process_message_receive(rtsp_engine_t *engine, rtsp_connection_t *connection, rtsp_message_t *message)
{
	rtsp_session_t *session = NULL;
	if(!engine || !message) {
		return FALSE;
	}

	if(message->start_line.message_type == RTSP_MESSAGE_TYPE_REQUEST) {
		if(message->start_line.common.request_line.method_id == RTSP_METHOD_SETUP) {
			if(!message->header.session_id) {
				/* create new rtsp session */
				session = rtsp_session_create(engine,TRUE);
				if(session){
					rtsp_session_connection_attach(engine,session,connection);
					rtsp_engine_session_table_add(engine,session);
					engine->handler->on_session_create(engine->handler,session);
				}
			}
		}
		else if(message->start_line.common.request_line.method_id == RTSP_METHOD_DESCRIBE) {
			if(!message->header.session_id) {
				/* create new rtsp session (will be used as communication handel only) */
				session = rtsp_session_create(engine,FALSE);
				if(session){
					rtsp_session_connection_attach(engine,session,connection);
				}
			}
		}

		if(!session && message->header.session_id) {
			session = rtsp_engine_session_find(engine,message->header.session_id,strlen(message->header.session_id));
		}
		if(session) {
			engine->handler->on_request_receive(engine->handler,session,message);
		}
		else {
			rtsp_message_t *response = rtsp_response_create(message,RTSP_STATUS_CODE_NOT_FOUND,RTSP_REASON_PHRASE_NOT_FOUND,message->pool);
			engine->connection_agent->method_set->send(connection,response);
		}
	}
	else if(message->start_line.message_type == RTSP_MESSAGE_TYPE_RESPONSE) {
		session = apr_hash_get(engine->request_queue,&message->header.cseq,sizeof(message->header.cseq));
		if(session) {
			rtsp_message_t *request;
			if(!session->id.hex_str && message->header.session_id) {
				session->id.hex_str = message->header.session_id;
				session->id.length = strlen(message->header.session_id);
				rtsp_engine_session_table_add(engine,session);
			}

			apr_hash_set(engine->request_queue,&message->header.cseq,sizeof(message->header.cseq),NULL);
			request = apt_ptr_queue_pop(session->queue);
			if(!request) {
				apt_log(APT_PRIO_WARNING,"Found no RTSP request waiting for response [%d]\n",message->header.cseq);
			}
			else if(request->header.cseq != message->header.cseq) {
				apt_log(APT_PRIO_WARNING,"Found mismatch in RTSP response [%d] and request [%d]\n",
					message->header.cseq, request->header.cseq);
			}
			engine->handler->on_response_receive(engine->handler,session,message,request);

			request = apt_ptr_queue_head(session->queue);
			if(request) {
				if(engine->connection_agent->method_set->send(session->connection,request) == TRUE) {
					apr_hash_set(engine->request_queue,&request->header.cseq,sizeof(request->header.cseq),session);
				}
			}
		}
		else {
			apt_log(APT_PRIO_WARNING,"Found no RTSP request waiting for response [%d]\n",message->header.cseq);
		}
	}
	return TRUE;
}

static apt_bool_t rtsp_engine_process_message_send(rtsp_engine_t *engine, rtsp_session_t *session, rtsp_message_t *message)
{
	if(session->id.hex_str) {
		message->header.session_id = session->id.hex_str;
		rtsp_header_property_add(&message->header.property_set,RTSP_HEADER_FIELD_SESSION_ID);
	}
	
	if(message->start_line.message_type == RTSP_MESSAGE_TYPE_REQUEST) {
		message->header.cseq = ++engine->last_cseq;
		rtsp_header_property_add(&message->header.property_set,RTSP_HEADER_FIELD_CSEQ);

		/* push message to session message queue */
		apt_ptr_queue_push(session->queue,message);
		if(apt_ptr_queue_head(session->queue) == message) {
			/* no other request is in a session queue, trying to send the request and 
			push entire session to waiting for response queue of the engine */
			if(engine->connection_agent->method_set->send(session->connection,message) == TRUE) {
				apr_hash_set(engine->request_queue,&message->header.cseq,sizeof(message->header.cseq),session);
			}
		}
	}
	else if(message->start_line.message_type == RTSP_MESSAGE_TYPE_RESPONSE) {
		engine->connection_agent->method_set->send(session->connection,message);
	}
	return TRUE;
}

static apt_bool_t rtsp_engine_process_session_create(rtsp_engine_t *engine, rtsp_session_t *session, rtsp_message_t *message)
{
	session->connection = engine->connection_agent->method_set->connect(engine->connection_agent);
	if(!session->connection) {
		return engine->handler->on_session_terminate(engine->handler,session);
	}

	rtsp_session_connection_attach(engine,session,session->connection);

	if(message) {
		rtsp_engine_process_message_send(engine,session,message);
	}
	return TRUE;
}

static apt_bool_t rtsp_engine_process_session_terminate(rtsp_engine_t *engine, rtsp_session_t *session)
{
	rtsp_session_connection_dettach(engine,session);
	rtsp_engine_session_table_remove(engine,session);
	return engine->handler->on_session_terminate(engine->handler,session);
}

static apt_bool_t rtsp_engine_process_peer_connect(rtsp_engine_t *engine, rtsp_connection_t *connection)
{
	connection->ref_count++;
	return TRUE;
}

static apt_bool_t rtsp_engine_process_peer_disconnect(rtsp_engine_t *engine, rtsp_connection_t *connection)
{
	connection->ref_count--;
	if(connection->ref_count > 0) {
		apr_hash_index_t *hi;
		void *val;
		rtsp_session_t *session;
		apt_log(APT_PRIO_NOTICE,"Destroy Remaining Sessions [%d]\n",connection->ref_count);
		for(hi=apr_hash_first(engine->pool,engine->session_table); hi;) {
			apr_hash_this(hi, NULL, NULL, &val);
			hi = apr_hash_next(hi);
			if(val) {
				session = val;
				if(session->connection == connection) {
					rtsp_engine_process_session_terminate(engine,session);
					if(connection->ref_count == 0) {
						break;
					}
				}
			}
		}
	}

	if(connection->ref_count > 0) {
		apt_log(APT_PRIO_WARNING,"Garbage Connection Detected: ref count [%d]\n",connection->ref_count);
		return FALSE;
	}

	engine->connection_agent->method_set->destroy_connection(connection);
	return TRUE;
}

static apt_bool_t rtsp_engine_msg_handler(void *object, apt_task_msg_t *task_msg)
{
	rtsp_engine_t *engine = object;
	rtsp_engine_msg_t *msg = (rtsp_engine_msg_t*) task_msg->data;
	assert(engine);

	switch(msg->id) {
		case RTSP_CONNECTION_MSG_AGENT_OPEN:
			break;
		case RTSP_CONNECTION_MSG_AGENT_CLOSE:
			break;
		case RTSP_CONNECTION_MSG_PEER_CONNECT:
			rtsp_engine_process_peer_connect(engine,msg->connection);
			break;
		case RTSP_CONNECTION_MSG_PEER_DISCONNECT:
			rtsp_engine_process_peer_disconnect(engine,msg->connection);
			break;
		case RTSP_CONNECTION_MSG_RECEIVE:
			rtsp_engine_process_message_receive(engine,msg->connection,msg->message);
			break;
		case RTSP_USER_MSG_SESSION_CREATE:
			rtsp_engine_process_session_create(engine,msg->session,msg->message);
			break;
		case RTSP_USER_MSG_SESSION_TERMINATE:
			rtsp_engine_process_session_terminate(engine,msg->session);
			break;
		case RTSP_USER_MSG_SEND:
			rtsp_engine_process_message_send(engine,msg->session,msg->message);
			break;
	}
	return TRUE;
}

static void rtsp_engine_on_start_in_progress(void *data)
{
	rtsp_engine_t *engine = data;
	assert(engine);

	apt_log(APT_PRIO_INFO,"Initialize RTSP Engine\n");

	if(engine->connection_agent) {
		engine->connection_agent->method_set->open(engine->connection_agent);
	}
}

static void rtsp_engine_on_terminate_in_progress(void *data)
{
	rtsp_engine_t *engine = data;
	assert(engine);

	if(engine->connection_agent) {
		engine->connection_agent->method_set->close(engine->connection_agent);
	}
}

static void rtsp_engine_on_terminate_complete(void *data)
{
	rtsp_engine_t *engine = data;
	assert(engine);

	apt_log(APT_PRIO_INFO,"Destroy RTSP Engine\n");

	if(engine->connection_agent) {
		engine->connection_agent->method_set->destroy(engine->connection_agent);
	}
}

static apt_bool_t rtsp_engine_on_agent_open(rtsp_connection_agent_t *agent)
{
	return rtsp_engine_connection_task_msg_signal(
										RTSP_CONNECTION_MSG_AGENT_OPEN,
										agent,
										NULL,
										NULL);
}

static apt_bool_t rtsp_engine_on_agent_close(rtsp_connection_agent_t *agent)
{
	return rtsp_engine_connection_task_msg_signal(
										RTSP_CONNECTION_MSG_AGENT_CLOSE,
										agent,
										NULL,
										NULL);
}

static apt_bool_t rtsp_engine_on_peer_connect(rtsp_connection_t *connection)
{
	return rtsp_engine_connection_task_msg_signal(
										RTSP_CONNECTION_MSG_PEER_CONNECT,
										connection->agent,
										connection,
										NULL);
}

static apt_bool_t rtsp_engine_on_peer_disconnect(rtsp_connection_t *connection)
{
	return rtsp_engine_connection_task_msg_signal(
										RTSP_CONNECTION_MSG_PEER_DISCONNECT,
										connection->agent,
										connection,
										NULL);
}

static apt_bool_t rtsp_engine_on_message_receive(rtsp_connection_t *connection, rtsp_message_t *message)
{
	return rtsp_engine_connection_task_msg_signal(
										RTSP_CONNECTION_MSG_RECEIVE,
										connection->agent,
										connection,
										message);
}

static apt_bool_t rtsp_engine_connection_task_msg_signal(
										rtsp_engine_msg_id       id,
										rtsp_connection_agent_t *agent,
										rtsp_connection_t       *connection,
										rtsp_message_t          *message )
{
	apt_task_msg_t *task_msg;
	rtsp_engine_msg_t *engine_msg;
	rtsp_engine_t *engine;

	if(!agent || !agent->event_handler) {
		return FALSE;
	}

	engine = agent->event_handler->object;
	if(!engine) {
		return FALSE;
	}

	task_msg = apt_task_msg_acquire(engine->msg_pool);
	if(!task_msg) {
		return FALSE;
	}

	task_msg->msg_pool = engine->msg_pool;
	task_msg->msg_handle = agent->event_handler;

	engine_msg = (rtsp_engine_msg_t*)task_msg->data;
	engine_msg->id = id;
	engine_msg->agent = agent;
	engine_msg->connection = connection;
	engine_msg->message = message;

	return apt_consumer_task_signal(engine->consumer_task,task_msg);
}

static apt_bool_t rtsp_engine_user_task_msg_signal(
										rtsp_engine_msg_id id,
										rtsp_session_t    *session,
										rtsp_message_t    *message )
{
	apt_task_msg_t *task_msg;
	rtsp_engine_msg_t *engine_msg;

	if(!session || !session->engine) {
		return FALSE;
	}

	task_msg = apt_task_msg_acquire(session->engine->msg_pool);
	if(!task_msg) {
		return FALSE;
	}

	task_msg->msg_pool = session->engine->msg_pool;
	task_msg->msg_handle = session->engine;

	engine_msg = (rtsp_engine_msg_t*)task_msg->data;
	engine_msg->id = id;
	engine_msg->session = session;
	engine_msg->message = message;

	return apt_consumer_task_signal(session->engine->consumer_task,task_msg);
}
